#!/bin/usr/sage 
r"""
This is program written in sage for rounding exact solutions from flag algebra computations.
Does not do everything, but can do something if you are lucky.
In general, you need to provide
 - SDP program  dat-s and its solution dat-s.result from CSDP
 - eigenvectors corresponding to zero eigenvalues of matrices in the solution.

NOTE: SDP program needs to have ALL entries integers. So before rounding, you need
to multiply the objective function or constraints accordingly. This is because the
rounding is done over integers to avoid rounding errors.

The rounding procedure is trying to guess graphs, where the CSDP program
must be tight. Usually, subgraphs of the extremal example. You may provide them
or hope the program can guess them right.

Few more notes:
 - solutions can be seen as matrices but also as one long vector
   during rounding, we look at the solution as a long vector.


cat altered.dat-s.solution | sed -n '/yMat/,$p' | sed 's/}$/],/g' | sed 's/}/]/g'| sed 's/{/[/g'  | grep -v '=' | sed 's/^ *\[ *$/yMat = \[/' | sed 's/^],$/]/' > altered.dat-s.solution.py
 
 Some description of the rounding procedure:
 Problem:
 csdp is solving problem like:
    max C.X
    subject to A_1.X = b_1
               A_2.X = b_2
               A_mdim.X = b_mdim
    where X is positive semidefinite matrix
 We actually solve it as
 max t 
 subject to 
    A_1.X = b_1
    A_2.X = b_2
 A_mdim.X = b_mdim
 where X is positive semidefinite matrix

   b_i + A_i X >= t
 
 //      max   -t
 //      s.t.  [A.X]_i + s_i - t = -D_i 
 

load('../rounding_exact.sage') 
p=FAProblem('SDP_n6_UB_F_3edges2__objective.txt.dat-s')
zv=[]
pp=p.change_base(zv)
write_problem_file('altered.dat-s',pp)

solve_using_csdp("altered.dat-s")
 
pa=FAProblem('altered.dat-s')
pa.perform_rounding(objective)

write_rounded_solution_to_file(pa)
 

uf=load_unlabeled_flags("F_vertices2_4edges__n7_unlabeled.txt",7)

[ uf[id] for id in pa.tightgraphsIDS ]

with open('alD_tight.txt', "wt") as file:
    for id in pD.tightgraphsIDS:
        file.write(uf[id]+'\n')


check_rounded_solution('altered.dat-s')

"""

import numpy
import subprocess
import sys
import pickle  # for saving/loading the rounded matrices
import itertools
import random
import shutil
import os.path

field=Integer

def rescale_number(x, constants_rescale, kill_if_fail=True):
    """ Rescaling and testing that rescaling indeed gives an integer"""
    
    if constants_rescale == 1:
        return field(float(x))
    
    x_float = float(x)*constants_rescale
    x_rounded = round(x_float)
    if abs(x_float - x_rounded) > 0.0000001:
        print("Rescaling did not work for x=",x," result was",x_rounded,"which is not precise for",x_float)
        print("Try a different rescaling of the semidefinite program.")
        if kill_if_fail:
            sys.exit()
 
    return field(x_rounded)
#field(int(float(parts[i])*constants_rescale+0.5))


################################################################################### UTILITIES
# Reads the whole problem as integer program
# the program has to be integral. This is a hack
# for SDP programs that were solved without scaled objective
# and are expensive to resolve
def read_problem_file(filename,constants_rescale=1,constraint_from=0, constraint_to=10000000):
    f = open(filename, 'r')

    line = f.readline()
    line = line.strip()
    num_constraints = int(line)

    line = f.readline()
    line = line.strip()
    num_blocks = int(line)

    line = f.readline()
    parts = line.split()
    block_sizes = []
    for part in parts:
        block_sizes.append(int(part))

    line = f.readline()
    parts = line.split()

    constants = []
    used_constraints=0

    for i in range(num_constraints):
        if i >= constraint_from and i < constraint_to:
            constants.append(rescale_number(parts[i], constants_rescale))
            used_constraints += 1
    """            
    if constants_rescale == 1:
        for i in xrange(num_constraints):
           if i >= constraint_from and i < constraint_to:
               constants.append(field(parts[i]))
               used_constraints += 1
    else:
        for i in xrange(num_constraints):
           if i >= constraint_from and i < constraint_to:
               constants.append(field(int(float(parts[i])*constants_rescale+0.5)))
               used_constraints += 1
    """

    # blocks that are integer numbers right away
    blocks = []
    for size in block_sizes:
        if size < 0:
            #block = [ matrix(ZZ,1,-size) for x in range(num_constraints+1) ]
            block = [ matrix(ZZ,1,-size) for x in range(used_constraints+1) ]
        else:
            block = [ matrix(ZZ,size,size, sparse=True) for x in range(used_constraints+1) ]
            #block = [ matrix(ZZ,size,size, sparse=True) for x in range(num_constraints+1) ]
            #block = [ matrix(QQ,size,size) for x in range(num_constraints+1) ]
        blocks.append(block)


    while True:
        line = f.readline()
        if len(line) == 0:
            break
        parts = line.split()
        if len(parts) == 0:
            continue
        constraint = int(parts[0])
        # Read also a bit arround to accommodate for mistakes..
        # Blocks are shifted by one
        #if constraint <= constraint_from or constraint > constraint_to:
        #    continue
        block_num = int(parts[1])
        row = int(parts[2])
        column = int(parts[3])
        value = rescale_number(parts[4], constants_rescale)
        '''
        if constants_rescale == 1:
            value = field(parts[4])
        else:
            value = field(float(parts[4])*constants_rescale)
        '''
        if block_sizes[block_num - 1] < 0:
            blocks[block_num - 1][constraint-constraint_from][0, column - 1] += value
        else:
            blocks[block_num - 1][constraint-constraint_from][row - 1, column - 1] += value
            if row != column:
                blocks[block_num - 1][constraint-constraint_from][column - 1, row - 1] += value

    problem = dict()
    problem['num_constraints'] = used_constraints
    problem['num_blocks'] = num_blocks
    problem['block_sizes'] = block_sizes
    problem['blocks'] = blocks
    problem['constants'] = constants
    f.close()
    return problem


def write_problem_file(filename, problem):
    f = open(filename, 'w')

    num_constraints = problem['num_constraints']
    num_blocks = problem['num_blocks']
    block_sizes = problem['block_sizes']
    blocks = problem['blocks']
    constants = problem['constants']

    f.write("%d\n" % num_constraints)
    f.write("%d\n" % num_blocks)
    for blockID, size in enumerate(block_sizes):
        if size != 0:
            f.write("%d " % size)
        else:
            print("  Empty block %d " % blockID)
            f.write("1 ")
    f.write("\n")

    for constant in constants:
        f.write("%.20g " % constant)
    f.write("\n")

    def print_row():
        if value != 0:
            f.write("%d %d %d %d %.20g\n" % (constraint, block_num + 1, row + 1, column + 1, value))

    for constraint in range(num_constraints + 1):
        for block_num in range(num_blocks):
            block_size = block_sizes[block_num]
            if block_size < 0:
                for column in range(-block_size):
                    row = column
                    #print 'Writing',block_num,column,constraint
                    value = blocks[block_num][constraint][0,column]
                    print_row()
            else:
                d = blocks[block_num][constraint].dict()
                #d = blocks[block_num][constraint].dict(copy=False)
                for entry in d:
                    row = entry[0]
                    column = entry[1]
                    if column < row:
                        continue
                    value = d[entry]
                    print_row()
                #                for row in xrange(block_size):
                #                    for column in xrange(row, block_size):
                #                        value = blocks[block_num][constraint][row,column]
                #                        print_row()

    f.close()



# This reads the solution file - but only the primal. Dual is being ignored
# and it would waste space anyway. The problem is read as real numbers - because
# we 
def read_solution_file(filename, problem):
    f = open(filename, 'r')

    num_constraints = problem['num_constraints']
    num_blocks = problem['num_blocks']
    block_sizes = problem['block_sizes']

    Y = numpy.zeros([num_constraints], float)
    line = f.readline()
    parts = line.split()
    for i in range(num_constraints):
        Y[i] = float(parts[i])

    Primal = []
    Dual = []


    # ignore Dual ############################ We work only with primal anyway

    for block_num in range(num_blocks):
        if block_sizes[block_num] < 0:
            Primal.append(matrix(RDF, 1, -block_sizes[block_num]))
            #Dual.append(numpy.zeros([1, -block_sizes[block_num]], float))
        else:
            Primal.append(matrix(RDF, block_sizes[block_num], block_sizes[block_num]))
            #Dual.append(numpy.zeros([block_sizes[block_num], block_sizes[block_num]], float))

    while True:
        line = f.readline()
        if len(line) == 0:
            break
        parts = line.split()
        if len(parts) == 0:
            continue
        constraint = int(parts[0])
        block_num = int(parts[1])
        row = int(parts[2])
        column = int(parts[3])
        value = float(parts[4])

        if constraint == 2:
            block = Primal[block_num - 1]
        else:
            continue
            #block = Dual[block_num - 1]

        if block_sizes[block_num - 1] < 0:
            block[0, column - 1] += value
        else:
            block[row - 1, column - 1] += value
            if row != column:
                block[column - 1, row - 1] += value


    solution = dict()
    solution['Y'] = Y
    solution['Primal'] = Primal
    solution['Dual'] = Dual
    return solution


def calculateIndexes(problem):
    indexes = [] # matrix containing indeces for variables
    blocks = len(problem['blocks']) 
    cnt = 0
    for b in range(blocks):
        blocksize = problem['block_sizes'][b]
        if blocksize == -problem['num_constraints']-1 or blocksize == -problem['num_constraints']-2:
            indexes.append(-1)
            continue
        if blocksize > 0:
            MS = MatrixSpace(QQ,blocksize)
            indexes.append(MS())
            for i in range(blocksize):
                for j in range(i,blocksize):
                     indexes[b][j,i] = indexes[b][i,j] = cnt
                     cnt = cnt+1
        if blocksize < 0:
            MS = MatrixSpace(QQ,1,-blocksize)
            indexes.append(MS())
            for i in range(-blocksize):
                indexes[b][0,i] = cnt
                cnt = cnt+1
    return indexes


def all_blocks_positive_definite(M, printstuff = True, QQField = QQ):
    success = True
    for i in range(len(M)):
        B = M[i]
        if B.is_square() and B.ncols() >= 1:
            if B.is_positive_definite():
                if printstuff:
                    if QQField == QQ:
                        print("Block",i,"is positive definite")
                            #print "min eigval is",float(min(B.eigenvalues())) # this takes long time
                    else:
                        print("Block",i,"is positive definite")
            elif B.is_positive_semidefinite():
                if printstuff:
                    if QQField == QQ:
                        print("Block",i,"is positive semidefinite")
                            #print "min eigval is",float(min(B.eigenvalues())) # this takes long time
                    else:
                        print("Block",i,"is positive semidefinite")
            else:
                if printstuff:
                    if QQField == QQ:
                        print("Block",i,"IS NOT POSITIVE DEFINITE, min eigval is",float(min(B.eigenvalues())))
                    else:
                        print("Block",i,"is IS NOT POSITIVE DEFINITE")
                success=False
        else:
            for j in range(B.ncols()):
                if B[0,j] < 0:
                    if printstuff: print("Block",i,"IS NOT POSITIVE DEFINITE, B[0,",j,"]=",B[0,j])
                    success=False
    return success

def checkSolution(problem, variables, QQField=QQ, indexes=[], testconstraints=[], verbose=false):
   """Evaluates problem (program) with variables. It works as a test if you want to see if your
   solution is any good.
   """
   num_constraints = problem['num_constraints']
   num_blocks = problem['num_blocks']
   block_sizes = problem['block_sizes']
   blocks = problem['blocks']
   constants = problem['constants']
   
   if len(indexes) == 0:
       indexes = calculateIndexes(problem)
   
   if len(testconstraints) == 0:
        testconstraints = list(range(num_constraints))
   
   constantsVector = vector(QQField,constants)
   constantsVector = vector(QQField,constantsVector)

   RoundingLinearSystemVariableNumber = len(variables)   
   densityVector = [0] * len(constants)
   for i in testconstraints:
      t_factor = 0 # to make sure it is loaded
      #OriginalProbelmMatrix = matrix(QQField, 1, RoundingLinearSystemVariableNumber, sparse=True)
      VariablesDotBlocks =  QQField(0);
      for index in range(num_blocks):          
          size = block_sizes[index]
          if size > 0:
                d = blocks[index][i+1].dict()
                for entry in d:
                    x = entry[0]
                    y = entry[1]
                    if x < y:
                        continue
                    if x == y:
                        #OriginalProbelmMatrix[0, indexes[index][x,y] ] = d[entry] # blocks[index][i+1][x,y]
                        VariablesDotBlocks += variables[ indexes[index][x,y] ] * d[entry]
                    else:
                        #OriginalProbelmMatrix[0, indexes[index][x,y] ] = 2*d[entry] # 2*blocks[index][i+1][x,y] 
                        VariablesDotBlocks += 2 * variables[ indexes[index][x,y] ] * d[entry]
          if size < 0 and -size != num_constraints+1 and -size != num_constraints+2:
              for x in range(-size):
                  #OriginalProbelmMatrix[0, indexes[index][0,x] ] = blocks[index][i+1][0,x]
                  VariablesDotBlocks += variables[ indexes[index][0,x] ] * blocks[index][i+1][0,x]
          else:
              if -size != num_constraints+1 or -size != num_constraints+2:
                  t_factor = abs(blocks[index][i+1][0,-1])
          
      #densityVector[i] =  constantsVector[i] - (OriginalProbelmMatrix * variables)[0]
      densityVector[i] =  (constantsVector[i] - VariablesDotBlocks)/t_factor
   #densityVector = constantsVector - OriginalProbelmMatrix * variables
   return densityVector


# Function to write Ping's solution 
def printLinear(linear,denominator,f=sys.stdout):
    f.write('denominatorLinear='+str(denominator)+'\n')
    f.write('solutionLinear=[')
    for i in range(len(linear)):
        if i != 0:
            f.write(', ')
        f.write(str(linear[i]*denominator))
    f.write(']')
    f.write('\n]')

def printMatrix(roundedMatrices,denominator):
    sys.stdout.write('denominatorLinear='+str(denominator)+'\n')
    sys.stdout.write('solutionLinear=[')
    for b in range(len(roundedMatrices)):
        if b != 0:
            sys.stdout.write(',\n')
        for i in range(len(roundedMatrices[b][0])):
            for j in range(i,len(roundedMatrices[b][0])):
                if i != 0 or j != 0: 
                   sys.stdout.write(', ')
                sys.stdout.write(str(roundedMatrices[b][i,j]*denominator))
    sys.stdout.write(']')
    sys.stdout.write('\n')

def write_rounded_solution_to_file(p, filename=None):
    if filename == None:
        filename = p.filename+".linear_solution.sage"
    f = open(filename, "w")
    f.write('solutionLinear=[')
    for i in range(len(p.solutionRounded)):
        if i != 0:
            f.write(', ')
        f.write(str( p.solutionRounded[i]))
    f.write(']')
    f.write('\n')  
    f.close()  


def load_rounded_solution_from_file(p = None, filename = None):
    if p != None and filename == None:
        filename = p.filename+".linear_solution.sage"
    load(filename)
    if p != None:
        p.solutionRounded = solutionLinear
    return solutionLinear

def check_rounded_solution(filename, optimal_value, constants_rescale=1, solution_filename=None):
    print("Loading problem and solution")
    p = FAProblem(filename, constants_rescale)
    linearSolution = load_rounded_solution_from_file(p, solution_filename)
    print("checking value of the solution")
    output = checkSolution(p.problem, linearSolution)
    if min(output) != optimal_value*constants_rescale:
        print(("Not optimal value! match for", min(output),"and",optimal_value*constants_rescale))
        return False
    print("Optimal value match. Checking if blocks are positive definite")
    # check positive definite
    matrixSolution = linearToMatrixP(p.problem, linearSolution)
    if all_blocks_positive_definite(matrixSolution) == False:
        return False

    print("All looks good")

    return True

def caculateDim(problem):
    dim = []  # may be negative 
    for i in range(len(problem['blocks'])):
        blocksize=problem['block_sizes'][i]
        # Slack variables
        if (blocksize == -problem['num_constraints']-1 or blocksize == -problem['num_constraints']-2):
            dim.append(0)
            continue
        dim.append(blocksize)
    return dim


def linearToMatrix(blocks,dim,linearSolution,QQField=QQ):
   M=[]
   cnt = 0
   for b in range(blocks):
       if dim[b] > 0:
           MS = MatrixSpace(QQField,dim[b])
           M.append(MS())
           for i in range(dim[b]):
               for j in range(i,dim[b]):
                   M[b][j,i] = M[b][i,j] = linearSolution[cnt]
                   cnt=cnt+1
       if dim[b] < 0:
           MS = MatrixSpace(QQField,1,-dim[b])
           M.append(MS())
           for i in range(-dim[b]):
                   M[b][0,i] = linearSolution[cnt]
                   cnt=cnt+1
       if dim[b] == 0:
           MS = MatrixSpace(QQField,0,0)
           M.append(MS())
 
   return M

def linearToMatrixP(problem,linearSolution,QQField=QQ):
    return linearToMatrix(len(problem['blocks']), caculateDim(problem), linearSolution, QQField)


def matrixToLinear(blocks,dim,M,QQField=QQ):
   variables = 0
   for b in range(blocks):
      variables += (dim[b]*(dim[b]+1))/2
   L=vector(QQField,int(variables))
   cnt = 0
   for b in range(blocks):
       for i in range(dim[b]):
           for j in range(i,dim[b]):
               L[cnt] = M[b][j,i]
               cnt=cnt+1
   return L


def guess_zero_eigenvectors(m):
    ''' Call as guess_zero_eigenvectors(p.solnMatrix[0])
    This is a fun utility that may work if the extremal construction
    is nicely balanced AND it is used on flags with k vertices and k-1 are labeled.
    It looks at zero eigenvectors, tries to find coordinates that are non-zero
    and in these coordinated it tries 0-1 vectors and sees if any are potentially
    zero vectors. Finally, it returns a basis of all these vectors.
    This is useful if there are possibly many ways to describe rooting or the
    extremal construction is not known.
    '''
    ev = m.eigenvectors_right()
    x= [ y for y in ev if y[0] < 0.000000001 ]
    b = sum( y[1][0] for y in x )
    z=vector([ round(100*y) for y in b ] )
    zID=vector([ y for y in range(len(z)) if z[y] != 0 ] )

    def makeVs(testV, zID, allV):
        if len(zID) == 0:
            #print(testV)
            allV.append(copy(testV))
            return

        id = zID[0]
        for value in [1,0]:
            testV[id] = value
            makeVs(testV, zID[1:], allV)

    testV = vector([ 0 for y in b ] )
    allV = []
    makeVs(testV, zID, allV)

    tightV = []
    for V in allV:
        if V*m*V < 0.0000001:
            tightV.append(vector(QQ, V))


    VV = QQ^len(z)
    zv = VV.span(tightV)

    b=zv.basis()
    return b

def round_vector(x):
    for z in x:
        if abs(round(z)-z) > 0.001:
            print("Coordinate",z,"may be wrong")
    return vector([ round(z) for z in x])

############################################################################ end of utilities

class FAProblem:
    """Contains problem and solution"""


    def __init__(self,filename,constants_rescale=1,QQField=QQ):
        """Initialization"""

        self.data = []
        self.problem=None
        self.solution=None
        self.solnMatrix=None
        self.solutionRounded=None

        self.QQField = QQField

        self.tightgraphs=[]
        self.tightgraphsIDS=[]
        self.extraTight=[]
        self.projectionMatrices=[]
        
        self.filename=filename
        
        self.constants_rescale=constants_rescale

        if filename != "":
            self.problem=read_problem_file(self.filename,self.constants_rescale)
            self.solution=read_solution_file(self.filename+".result",self.problem)
            self.solnMatrix = self.solution['Primal']

        self.tightgraphsIDS = None

    def reinit(self, problem, solution):
        """Initialization"""
        
        self.data = []
        self.problem=problem
        self.solution=solution
        if solution != None:
            self.solnMatrix=self.solution['Primal']
        else:
            self.solnMatrix=[]
        self.solutionRounded=None
        
        self.tightgraphs=[]
        self.tightgraphsIDS=[]
        self.extraTight=[]
        self.projectionMatrices=[]
        
        self.filename="unknown"

        self.tightgraphsIDS = None


    def test_ranks(self):
        """Testing ranks of matrices in the solution"""
        block_sizes=self.problem['block_sizes']
        nextBlock = 0
        for idA in range(len(self.solnMatrix)):
            # These are slack variables - do not care about these guys
            if -block_sizes[idA] == self.problem['num_constraints']+1 or -block_sizes[idA] == self.problem['num_constraints']+2:
                continue
            A = self.solnMatrix[idA]
            # Do not bother with matrices that are not squares
            # they do not need any special attention
            if not A.is_square():
                continue
            # Now test rank of A
            print("Block",idA,"of size",A.ncols(),"has rank",A.rank())



    def eigenvalues(self, zv=None, zero=0.0000001):
        """Prints smallest eigenvalue for every block of solution."""
        #global problem
        #global solutionMatrix
        block_sizes=self.problem['block_sizes']
        #nextBlock = 0
        all_positive_definite = True
        for idA in range(len(self.solnMatrix)):
            # These are slack variables - do not care about these guys
            if -block_sizes[idA] == self.problem['num_constraints']+1 or -block_sizes[idA] == self.problem['num_constraints']+2:
                continue
            A = self.solnMatrix[idA]
            # Do not bother with matrices that are not squares
            # they do not need any special attention
            if not A.is_square():
                all_entries=[]
                for j in range(A.ncols()):
                    all_entries.append(A[0,j])
                if min(all_entries) < zero: 
                    print("Diagonal block",idA,"HAS zero entries, minimum is ",min(all_entries))
                    #all_positive_definite = False
                else:
                    print("Diagonal block",idA,"is positive definite ",min(all_entries))
                continue
            # Now we try to find all eigenvalues that are small
            zero_eigenvalues=[]
            smallest_ev = -1
            for ev in A.eigenvectors_right():
                if (smallest_ev == -1 or abs(ev[0]) < smallest_ev):
                    smallest_ev = abs(ev[0])
                if (abs(ev[0]) < zero):
                    zero_eigenvalues.append(abs(ev[0]))
            if smallest_ev > zero:
                if zv != None and len(zv[idA]) != 0:
                    print("Block",idA,"is positive definite, provided ",matrix(zv[idA]).rank() ," zero vectors, smalest eigenvalue is ",smallest_ev)
                else:
                    print("Block",idA,"is positive definite ",smallest_ev)
            else:
                if zv != None and len(zv[idA]) != 0:
                    print("Block",idA,"has ",len(zero_eigenvalues)," zero eigenvalues, provided ",matrix(zv[idA]).rank() ," zero vectors, smallest ev is",smallest_ev)
                    if len(zero_eigenvalues) > matrix(zv[idA]).rank():
                        all_positive_definite = False
                else:
                    print("Block",idA,"has ",len(zero_eigenvalues)," zero eigenvalues, smallest is",smallest_ev)
                    all_positive_definite = False
        return all_positive_definite

    # Prints ten smallest eigenvalues in block idA
    # This is usefull 
    def test_block(self,idA):
        block_sizes=self.problem['block_sizes']
        A = self.solnMatrix[idA]
        if not A.is_square():
            print("Not a square matrix")
            return
        X = sorted([ real_part(ev[0]) for ev in A.eigenvectors_right()])
        print(X[0:min(len(X),10)])
        return X







    # Try to identify tight graphs
    # This is a sanity check - should not be changing when removing types
    # returns lists that contain 3 elemets:
    # 1... value of slack variable.  Slack = 0 means tight graph
    # 2... value in dual. Dual >= means tight graph (appears in extremal construction)
    # 3... ID in order of the graph
    # can be sorted using
    #  sa=sorted(x, key = lambda y: y[0])
    #  sa=sorted(x, key = lambda y: y[1])
    #
    def guess_tight_graphs(self,threshold = 0.000001):
        block_sizes=self.problem['block_sizes']
        self.tightgraphs=[]
        self.tightgraphsIDS=[]
        tightlist=[]
        # test if zero in primal vector
        for idA in range(len(self.solnMatrix)):
            # These are slack variables - do not care about these guys
            if -block_sizes[idA] == self.problem['num_constraints']+1 or -block_sizes[idA] == self.problem['num_constraints']+2:
                A = self.solnMatrix[idA] 
                # Notice that solnMatrix[idA][0,0] contains the extra variable for
                # objective function. So the first slack is at solnMatrix[idA][0,1]
                for i in range(self.problem['num_constraints']):
                    tightlist.append([A[0,i],self.solution['Y'][i],i])
                    #tightlist.append([A[0,i+1],self.solution['Y'][i],i])
                    #tightlist.append([A[0,i+1],solution['Y'][i],density_in_blowup(i),i])
                    if A[0,i] < threshold:
                        print('tight',i,tightlist[i])
                        self.tightgraphs.append(1)
                        self.tightgraphsIDS.append(i)
                    else:
                        self.tightgraphs.append(0)
                        print('loose',i,tightlist[i])
                print("idA=",idA)
                break
        return tightlist
        # test if non-zero in dual vector. It is slightly unprecise
        # as it only works for complementary slackness
        '''
        for i in range(len(solution['Y'])):
            if solution['Y'][i] > 0.0000001:
                tightgraphs.append(1)
                tightgraphsIDS.append(i)
            else:
                tightgraphs.append(0)
        '''



    # Copies problem and does a base change based on zero vectors.
    # Zero vectors are vectors we believe are zero eigenvectors of
    # blocks in the problem matrix. If we find them all, it would
    # be possible that CSDP returns to us a matrix that is positive
    # definite instead of positive semidefinite and it becomes much
    # easier to round.
    #
    def change_base(self,zerovectors,printstuff=True,inPlace=False):
        #global problem
        #global solutionMatrix
        if printstuff:
            print("Making a copy")
        if inPlace:
            newproblem = self.problem
        else:
            newproblem = deepcopy(self.problem);
        block_sizes=self.problem['block_sizes']
        nextBlock = 0
        self.projectionMatrices=[]
        if printstuff:
            print("Processing")
        #for idA in range(len(self.solnMatrix)):
        for idA in range(len(block_sizes)):
            if -block_sizes[idA] == self.problem['num_constraints']+1 or -block_sizes[idA] == self.problem['num_constraints']+2:
                continue
            #A = self.solnMatrix[idA]
            # Do not bother with matrices that are not squares
            # they do not need any special attention
            # Or if they are just constants...
            #if not A.is_square() or (1 == A.nrows() == A.ncols()):
            if block_sizes[idA] < 0:
                continue

            if  block_sizes[idA] == 1 or block_sizes[idA] == 0:
                self.projectionMatrices.append(None)
                nextBlock+=1
                continue

            if len(zerovectors) < nextBlock:
                print("Not enought zero vectors")
                return
            if len(zerovectors[nextBlock]) > 0:

                A = self.solnMatrix[idA]
                zeros = sum([ 1 for  ev in A.eigenvalues() if ev < 0.00000001 ])

                if printstuff: 
                    print("Changing base of block",idA,"of size",newproblem['block_sizes'][idA],"using",len(zerovectors[nextBlock]),"eigenvector(s) for zero. May need ",zeros,'zero vectors')
                
                # Here we want matrix with integer entries
                #V = Matrix(self.QQField, zerovectors[nextBlock],sparse=True)
                # Integers make nicer projection matrices without fractions
                V = Matrix(ZZ, zerovectors[nextBlock],sparse=True)

                #print("nextBloxk=",nextBlock,"idA=",idA)
                #print("Zeros",zerovectors[nextBlock])

                # Orthogonal complement of our eigenvector
                RestBase = V.right_kernel_matrix().sparse_matrix()

                self.projectionMatrices.append(RestBase)
                
                #print "Base orthogonal to zerovector: T="
                #print RestBase
                #print RestBase*A*RestBase.transpose()

                #print("idA=",idA)
                #print("Rest base=",RestBase)
                for idX in range(len(newproblem['blocks'][idA])):
                    #print("idX=",idX)
                    newproblem['blocks'][idA][idX] = RestBase*newproblem['blocks'][idA][idX]*RestBase.transpose()
                newproblem['block_sizes'][idA] = newproblem['blocks'][idA][idX].nrows()
        
                if printstuff: print("Block",idA,"has size",newproblem['blocks'][idA][idX].nrows())
                #newproblem['block_sizes'][idA] -= len(zerovectors[nextBlock])
       

                #normZ = vector(zerovector).norm()
                #normalZ = [ float(y/normZ) for y in zerovector ]
                #print "Normalized eigenvector for zero:",normalZ
                #T = Matrix([[ float(y) for y in z/z.norm()] for z in RestBase.rows()]+[normalZ])
                #T = Matrix(RestBase.rows()+[zerovector])
                #print "Matrix T*A*T^T="
                #print T*A*T.transpose()
                #X = Matrix(RestBase.rows()+[zerovector]).transpose()
                #print X
                #print X.inverse()*A*X
            else:

                A = self.solnMatrix[idA]
                zeros = sum([ 1 for  ev in A.eigenvalues() if ev < 0.0000001 ])

                print("Block",idA,"no zero eigenvectors provided, may need",zeros,"of them")
                self.projectionMatrices.append(None)
            nextBlock+=1
            #print "Done with",idA
        return newproblem

# Copies problem and does a base change based on zero vectors.
# Zero vectors are vectors we believe are zero eigenvectors of
# blocks in the problem matrix. If we find them all, it would
# be possible that CSDP returns to us a matrix that is positive
# definite instead of positive semidefinite and it becomes much
# easier to round.
#
    def change_base_using_projectionMatrices(self,projectionMatrices,printstuff=True,inPlace=False):
        if printstuff:
            print("Making a copy")
        if inPlace:
            newproblem = self.problem
        else:
            newproblem = deepcopy(self.problem);
        block_sizes=self.problem['block_sizes']
        self.projectionMatrices = projectionMatrices
        nextBlock = 0
        if printstuff:
            print("Processing")
        for idA in range(len(self.solnMatrix)):
            if -block_sizes[idA] == self.problem['num_constraints']+1 or -block_sizes[idA] == self.problem['num_constraints']+2:
                continue
            A = self.solnMatrix[idA]
            # Do not bother with matrices that are not squares
            # they do not need any special attention
            # Or if they are just constants...
            #if not A.is_square() or (1 == A.nrows() == A.ncols()):
                #newproblem['blocks'] = newproblem['blocks']+[problem['blocks'][idA]]
            #    continue

            if block_sizes[idA] < 0:
                continue

            if  block_sizes[idA] == 1 or block_sizes[idA] == 0:
                nextBlock+=1
                continue

            if projectionMatrices[idA] != None:
                if printstuff: print("Changing base of block",idA,"of size",newproblem['block_sizes'][idA],"using prelodaded projection matrix")
                #V = Matrix(zerovectors[nextBlock],sparse=True)
                # Orthogonal complement of our eigenvector
                #RestBase = V.right_kernel_matrix().sparse_matrix()
                #self.projectionMatrices.append(RestBase)
                RestBase = projectionMatrices[idA]
                
                for idX in range(len(newproblem['blocks'][idA])):
                    newproblem['blocks'][idA][idX] = RestBase*newproblem['blocks'][idA][idX]*RestBase.transpose()
                newproblem['block_sizes'][idA] = newproblem['blocks'][idA][idX].nrows()
                
                if printstuff: print("Block",idA,"has size",newproblem['blocks'][idA][idX].nrows())
            else:
                print("Block",idA,"has no projection.")
            nextBlock+=1
        return newproblem


    def change_base_using_projectionMatricesAsLists(self,projectionMatricesAsList,printstuff=True,inPlace=False):
        projectionMatrices = []
        for ML in projectionMatricesAsList:
            if len(ML) == 0:
               projectionMatrices.append(None)
            else :
                projectionMatrices.append(matrix(ML, sparse=True))
        return  self.change_base_using_projectionMatrices(projectionMatrices,printstuff,inPlace)




    def perform_rounding(self,optimal_value,zerovectors=[],QQField=QQ,printstuff=True,denominator=1000000000, save_solution=True):
        # Zerovectors may contain for each block a list of zero eigenvectors
        # QQField is usefull when used as quarticfield
        
        success=True        
        
        # Try to identify tight graphs
        if self.tightgraphsIDS == None:
            tightgraphs = []
            tightgraphsIDS = []
            for i in range(len(self.solution['Y'])):
                if self.solution['Y'][i] > 0.0000001 or i in self.extraTight:
                    tightgraphs.append(1)
                    tightgraphsIDS.append(i)
                else:
                    #print 'Not tight:',solution_alt['Y'][i]
                    tightgraphs.append(0)
        else:
            tightgraphsIDS = self.tightgraphsIDS
            tightgraphs = [0]*self.solution['Y']
            for t in tightgraphsIDS:
                tightgraphs[t] = 1
        self.tightgraphsIDS = tightgraphsIDS
        
        numtight = len(tightgraphsIDS)
        if printstuff: print("Guessed",numtight,"tight graphs")

 
        blocks = len(self.problem['blocks']) 

        dim = []  # may be negative 
        variables = 0


        for i in range(blocks):
            blocksize=self.problem['block_sizes'][i]
            # Slack variables
            if blocksize == -self.problem['num_constraints']-1 or blocksize == -self.problem['num_constraints']-2:
                dim.append(0) # indicates the objetive function
                continue
            if (blocksize > 0):
                dim.append(self.problem['block_sizes'][i])
                variables += dim[i]*(dim[i]+1)/2
            else:
                dim.append(self.problem['block_sizes'][i])
                variables += -dim[i]

        if printstuff: print("Problem has",variables,"variables")

        zero_constraints=sum([ len(b)*len(b[0]) for b in zerovectors if len(b) > 0  ]  )
        #zero_constraints=sum([ len(b) for b in zerovectors   ]  )
        if printstuff: print("Provided additional",zero_constraints,"zero constraints")
        
        constraints = numtight+zero_constraints
        if printstuff: print("Total number of constraints is",constraints)

       
        '''
        indexes = [] # matrix containing indeces for variables
        cnt = 0
        for b in range(blocks):
            #print dim[b]
            if dim[b] > 0:
                MS = MatrixSpace(QQ,dim[b])
                indexes.append(MS())
                for i in range(dim[b]):
                    for j in range(i,dim[b]):
                        indexes[b][j,i] = indexes[b][i,j] = cnt
                        cnt = cnt+1
            if dim[b] < 0:
                MS = MatrixSpace(QQ,1,-dim[b])
                indexes.append(MS())
                for i in range(-dim[b]):
                    indexes[b][0,i] = cnt
                    cnt = cnt+1
            if dim[b] == 0:
                indexes.append(-1)
        '''
        if printstuff: print("Creating indexes")
        indexes = calculateIndexes(self.problem)

        # Linearize the solution
        if printstuff: print("Linearizing solution")
        solutionX = vector(QQField, int(variables))
        cnt = 0
        for b in range(blocks):
            if dim[b] > 0:
                for i in range(dim[b]):
                    for j in range(i,dim[b]):
                        solutionX[cnt] = QQ(self.solution['Primal'][b][i,j])
                        cnt=cnt+1
            if dim[b] < 0:
                for i in range(-dim[b]):
                        solutionX[cnt] = QQ(self.solution['Primal'][b][0,i])
                        cnt=cnt+1
            #if dim[b] == 0:
            #    t_factor = 

        self.solutionX = solutionX
        
        # ROUND
        if printstuff: print("Testing the linear solution")        
        self.densityVectorX = checkSolution(self.problem, solutionX, QQField, indexes) 
        if printstuff:
            if QQField == QQ:
                print('Min of density vector for original solution is', float(min(self.densityVectorX)),'and we hope for',float(optimal_value*self.constants_rescale))
            else:
                print('Min of density vector for original solution is', min(self.densityVectorX),'and we hope for',optimal_value*self.constants_rescale)
            original=float(min(self.densityVectorX))
            expected=float(optimal_value*self.constants_rescale)
            if abs(original-expected) > 0.000001:
                print("ERROR: The difference of the original solution and expected one is too large.")
                if abs(abs(original)-abs(expected)) < 0.000001:
                    print("Looks like you made a sign error in the expected value of the solution.")
                if printstuff: print("Rounding failed")
                return False
        


        if printstuff: print("Creating and testing entry-wise rounded solution")
        #   Rounding the solution:
        solutionRounded = vector(QQField,len(solutionX))
        for i in range(len(solutionX)):
            solutionRounded[i] = floor(solutionX[i]*denominator)
            solutionRounded[i] /= denominator
        self.solutionRounded = solutionRounded
        
        
        self.densityVectorR = checkSolution(self.problem, solutionRounded, QQField, indexes) 
        if printstuff: 
            if QQField == QQ:
                print('Min of denisty just rounded vector is', float(min(self.densityVectorR)),'and we hope for',float(optimal_value*self.constants_rescale))
            else:
                print('Min of denisty just rounded vector is', min(self.densityVectorR),'and we hope for',optimal_value*self.constants_rescale)

        
        solutionRounded = solutionX
        
        #print 'Max of denisty vector is', float(max(densityVector))
        #print [ float(z) for z in densityVector ]

        #return  [self.densityVectorX,self.densityVectorR]

        if printstuff: print("Constructing a matrix A of constriants for tight graphs")
        MS = MatrixSpace(QQField,constraints,variables,sparse=False)
        #A = MS(sparse=False)  # zero matrix
        A = MS()  # zero matrix
        #AR = matrix(RDF, constraints, variables)  # zero matrix
        Y = vector(QQField,constraints)

        # constraints for tight graphs:
        # Note that the first g=0 is the objective function
        next_constraint = 0
        for g in range(self.problem['num_constraints']):
            if tightgraphs[g] == 1:
                t_factor = None
                for b in range(blocks):
                    if dim[b] > 0:
                        #'''
                        d = self.problem['blocks'][b][g+1].dict()
                        for entry in d:
                            row = entry[0]
                            column = entry[1]
                            if column < row:
                                continue
                            value = d[entry]
                            if row == column:
                                A[next_constraint, indexes[b][row, column]] = value
                            else:
                                A[next_constraint, indexes[b][row, column]] = 2*value
                            #AR[next_constraint, indexes[b][row, column]] = value
                        '''
                        for i in range(dim[b]):
                            for j in range(dim[b]):
                                if i == j:
                                    A[next_constraint, indexes[b][i,j]] = self.problem['blocks'][b][g+1][i,j]
                                else:
                                    A[next_constraint, indexes[b][i,j]] = 2*self.problem['blocks'][b][g+1][i,j]
                        '''
                    if dim[b] < 0:
                        for i in range(-dim[b]):
                            A[next_constraint, indexes[b][0,i]] = self.problem['blocks'][b][g+1][0][i]
                            #AR[next_constraint, indexes[b][0,i]] = self.problem['blocks'][b][g+1][0][i]
                    if dim[b] == 0:
                        t_factor = abs(self.problem['blocks'][b][g+1][0,-1])
                Y[next_constraint] = self.problem['constants'][g]-(optimal_value*self.constants_rescale*t_factor)  # 1 is the optimal solution
                next_constraint += 1

        #Zero constraints
        for b in range(len(zerovectors)):
            for v in zerovectors[b]:
                    if dim[b] > 0:
                        for i in range(dim[b]):
                            for j in range(dim[b]):
                                    A[next_constraint, indexes[b][i,j]] = v[j]
                                    #AR[next_constraint, indexes[b][i,j]] = v[j]
                            Y[next_constraint] = 0  
                            next_constraint += 1
                    if dim[b] < 0:
                        print("ERROR - untested") 
                        for i in range(-dim[b]):
                            A[next_constraint, indexes[b][0,i]] = v[0]
                            #AR[next_constraint, indexes[b][0,i]] = v[0]
                            Y[next_constraint] = 0  
                            next_constraint += 1
        
        if printstuff: print("A constructed of size",A.nrows(),"x",A.ncols())                      

        # Now matrix A should not be modified any more
        # it will lead to A.rank() being fast once computed once
        A.set_immutable()
        #AR.set_immutable()
        if printstuff: print("Rank of A is", A.rank())
        #if printstuff: print "A=",A

        # This is for experiments later
        self.A = A 


        # Finding linearly independent columns
        '''
        if printstuff: print "Calculating pivots of AR"
        pivots=AR.pivots()
        if printstuff: print "Pivots are",pivots
        '''
        if printstuff: print("Calculating pivots of A")
        pivots=find_pivots(A,QQField=QQField)
        self.pivots = pivots
        print("Pivots are", pivots)

        '''
        global main_pivots
        pivots=main_pivots
        '''

        test_pivots(A, pivots, QQField=QQField)

        MS = MatrixSpace(QQField,constraints,len(pivots))
        X = MS()
        use_presolved = []

        for p in range(len(pivots)):
            X[:,p] = A[:,pivots[p]]

        YforX = copy(Y)
        for i in range(variables):
            if not i in pivots:
                for j in range(constraints):
                    YforX[j] -= A[j,i]*solutionRounded[i]

        self.X = X
        self.Y = Y
        self.YforX = YforX
        if printstuff: print("Solving for pivot variables")
        not_presolved=X.solve_right(YforX)


        # Round all that are not presolved
        if printstuff: print("Differences in solved variables compared to rounded values (should be small)")
        for p in range(len(pivots)):
            if printstuff: print("Difference:",float(solutionRounded[pivots[p]]-not_presolved[p]))
            solutionRounded[pivots[p]] = not_presolved[p]


        if printstuff: print("Checking the FINAL solution")
        densityVector = checkSolution(self.problem, solutionRounded,QQField,indexes) 
        if printstuff:
            print('Min of denisty vector is', float(min(densityVector)))
            print('which is exact', min(densityVector))
            print(' and target is',optimal_value*self.constants_rescale)
        #print 'Max of denisty vector is', float(max(densityVector))
        self.solutionRoundedValue=float(min(densityVector))

        # These are the variables that violate the constriants
        violators=[]
        for i in range(len(densityVector)):
            if densityVector[i] < optimal_value*self.constants_rescale:
                violators.append(i)

        if len(violators) == 0:
            if printstuff: 
                print("Rounding successful!!")
                print("Solution is tight on the following")
                print([ i for i in range(len(densityVector)) if densityVector[i] == optimal_value*self.constants_rescale])
        else:
            if printstuff: print("Violated constriants:",violators)
            success=False


        if save_solution:
            write_rounded_solution_to_file(self)

        if printstuff: print("Testing if the rounded solution is positive definite")
        M=linearToMatrix(blocks,dim,solutionRounded,QQField)
        self.solutionRounded = solutionRounded
        self.solutionRoundedM = M
        
        if all_blocks_positive_definite(M, printstuff, QQField) == False:
            success = False

        if success:
            if printstuff: 
                print("Congratulations! Rounding succeeded")
        else:
            if printstuff: print("Rounding failed")


        return success

    # zero vectors should correspond to the zero eigenvectors
    # we do a test by multiplying the corresponding vectors and matrices
    # and see what is happening.
    def test_zerovectors(self, zerovectors, verbose=True):
        overall_max=None
        block_sizes=self.problem['block_sizes']
        nextBlock = 0
        for idA in range(len(self.solnMatrix)):
            if -block_sizes[idA] == self.problem['num_constraints']+1 or -block_sizes[idA] == self.problem['num_constraints']+2:
                continue
            A = self.solnMatrix[idA]
            # Do not bother with matrices that are not squares
            # they do not need any special attention
            if not A.is_square():
                continue
        
            if len(zerovectors) <= nextBlock:
                print("Not enought zero vectors, missing for block",nextBlock)
                return
            if len(zerovectors[nextBlock]) > 0:
                for v in range(len(zerovectors[nextBlock])):
                    Av = A*vector(zerovectors[nextBlock][v])
                    if verbose:
                        print('Block',idA,"provided zero vector id",v,"gives when mutliplied with block  max value",max([max(Av),-min(Av)]))
                if overall_max == None or overall_max <  max([max(Av),-min(Av)])   :
                    overall_max = max([max(Av),-min(Av)])
            #else:
            #    print "Block",idA,"has no zero eigenvalues."
            nextBlock+=1
        return overall_max



#unlabeled_flags=[]
# requires one flag per line. Nice in order.
def load_unlabeled_flags(fname, fsize):
    #global unlabeled_flags
    unlabeled_flags=[]
    with open(fname) as f:
        content = f.readlines()
        # remove whitespace characters like `\n` at the end of each line
        content = [x.strip() for x in content]
        for s in content:
            if int(s[0]) == fsize:
                unlabeled_flags.append(s)
    return unlabeled_flags



# Computes density of a graph in blow-up. This is somewhat slow
# as it is using external flag program
def density_in_blowup(fID):
    global unlabeled_flags
    global blow_up
    data=subprocess.Popen("./a.out -FinbupH '{}'  '{}' 2>/dev/null".format(unlabeled_flags[fID], blow_up),shell=True,stdout=subprocess.PIPE).communicate()[0].strip()
    return float(data[9:])

# This is good for getting a list of graphs that should be tight
def tight_in_blowup(unlabeled_flags, blow_up):
    tight_graphs=[]
    for i in range(len(unlabeled_flags)):
        result=subprocess.Popen("./a.out -eFinbupH '{}'  '{}' 2> /dev/null | sed 's/F in blowup of H: //'".format(unlabeled_flags[i], blow_up),shell=True,stdout=subprocess.PIPE).communicate()[0][0]
        if result == '1':
            tight_graphs.append(i)
    return tight_graphs

# Tests if pivots are indeed pivots for matrix A
# Just take the submatrix with the right columns and check the rank
def test_pivots(A, pivots, QQField=QQ):
    if A.rank() != len(pivots):
        print("Expected that",A.rank(),'==',len(pivots))
        

    B = matrix(QQField, A.nrows(), len(pivots))
    for row in range(A.nrows()):
        for column in range(len(pivots)):
            B[row,column] = A[row,pivots[column]]
    rank = B.rank()
    if rank != A.rank():
        print("Wrong rank",rank,'expected',A.rank())
        return False

    #Now test each pivot separately
    
    return True

def create_B(A, pivots, QQField=QQ):
    B = matrix(QQField, A.nrows(), len(pivots)+1)
    for row in range(A.nrows()):
        for column in range(len(pivots)):
            B[row,column] = A[row,pivots[column]]
    return B

def find_pivots_OLD(A,printstuff=False):
    pivots=[0]
    goalRank = A.rank() 
    B = create_B(A, pivots)
    for c in range(1,A.ncols()):
        if printstuff: print("Testing column",c,"out of",A.ncols(),"found",len(pivots),"so far, goal is",goalRank)
        lastB = B.ncols()-1
        for row in range(A.nrows()):
            B[row,lastB] = A[row,c]
        if B.rank() == B.ncols():
            pivots.append(c)
            if len(pivots) == goalRank:
                break
            B  = create_B(A, pivots)
    return pivots

def find_pivots(A,printstuff=False,QQField=QQ):
    pivots=[]
    goalRank = A.rank() 
    B = create_B(A, pivots, QQField)
    for c in range(0,A.ncols()):
        if printstuff: print("Testing column",c,"out of",A.ncols(),"found",len(pivots),"so far, goal is",goalRank)
        lastB = B.ncols()-1
        for row in range(A.nrows()):
            B[row,lastB] = A[row,c]
        if B.rank() == B.ncols():
            pivots.append(c)
            if len(pivots) == goalRank:
                break
            B  = create_B(A, pivots, QQField)
    return pivots

def solve_using_csdp(filename, csdp="csdp", omp=0):
    csdp_list=[ "./csdp", "~/csdp", "~/bin/csdp", "/usr/local/bin/csdp", "/usr/bin/csdp", "/usr/local/csdp/bin/csdp"]

    csdp_path = None

    if shutil.which(csdp):
        csdp_path = csdp
    for csdp_candidate in csdp_list:
        if shutil.which(csdp_candidate) or os.path.exists(csdp_candidate):
            csdp_path = csdp_candidate
            break

    if csdp_path is None:
        print("CSDP not found in path. Please install CSDP and add it to your PATH.")
        csdp_path=csdp

    #csdp="~/bin/csdp"
    if omp != 0:
        omp_str = "export OMP_NUM_THREADS="+str(omp)+"; "
    else:
        omp_str = ""
    print(subprocess.Popen(omp_str+csdp_path+" "+filename+" "+filename+".result | tee "+filename+".log",shell=True,stdout=subprocess.PIPE).communicate()[0])


def rounding_help():
    print("This program helps create rounding")

def resolve(pa, filename='altered.dat-s'):
    write_problem_file(filename,pa)
    solve_using_csdp(filename)
    return FAProblem(filename)



def solve_using_LP(p, it=20):

    num_blocks = p.problem['num_blocks']
    block_sizes = p.problem['block_sizes']
    blocks = p.problem['blocks']
    constants = p.problem['constants']
    num_constraints = p.problem['num_constraints']

    d = MixedIntegerLinearProgram(maximization=True)
    y = d.new_variable(real=True, nonnegative=True)  # will be used as an array of length constants 

    yrange = range(num_constraints)

    #print("Objective:",[ -c*y[i]  for i,c in enumerate(constants) ])

    d.set_objective( sum([ -c*y[i]  for i,c in enumerate(constants) ]) )

    d.add_constraint(sum([ y[i]  for i in yrange ])  == 1)
                            
    #d.add_constraint( 2.1708203932499366 *y1 +  -0.1708203932499368 *y2 +  -0.6180339887498947 *y3 >= 0 )

    dconstraints = 1 

    for test_run in range(it):

        print("Iter:",test_run+1,end=" ")
        print("Upper bound is",d.solve()/p.constants_rescale," with ",dconstraints," constraints")

        yval=[ d.get_values(y[i]) for i in  yrange ]

        #print("  yval=",yval)


        # This should run in parallel
        for blockID in range(num_blocks):
            if block_sizes[blockID] <= 0:
                continue

            A=blocks[blockID][0] # just zero matrix of the right size

            for i,yvali in enumerate(yval):
                A += yvali*blocks[blockID][i+1]

            #print("Matrix A")
            #print(A)
            A = matrix(RDF, A,sparse=False)
            evs = A.eigenvectors_right()
            #print("Got eigenvectors",evs)

            for e in evs:
                if e[0] < -0.00000001:
                    #print("  Eigenvalue",e[0],"with eigenvector",e[1][0])
                    v = e[1][0]

                    #print("  v=",v)

                    # Test if the new constraint violates the current solution
                    c = [ v*blocks[blockID][i+1]*v for i in yrange ]  

                    #print("c=",c)
                    #print("yval=",yval,"yrange=",yrange)

                    #print("presum=",[ c[i]*yval[i] for i in yrange])

                    # Test if we have an interesting constraint
                    new_contraint = sum([ real_part(c[i])*yval[i] for i in yrange])

                    #print("Sum=",new_contraint)

                    if new_contraint < 0:
                        #print("  Found new constraint",c)
                        # This is somewhat a critical section of the code.
                        d.add_constraint(sum([  real_part(c[i])*y[i]  for i in yrange ])  >= 0)
                        dconstraints += 1



def r(v):
    rounded= vector([round(x.real()) for x in v])
    if abs(rounded-v) > 0.0001:
        print('abs of r = ',abs(rounded-v))
    return rounded

def roundable_error(v):
    rounded= vector([round(x.real()) for x in v])
    return abs(rounded-v)

def roundable(v):
    return roundable_error(v) < 0.01

def guess_zero_vectors_eigenvectos_by_rounding(pa, zero=1.00000000000000e-7):
    zv=[ [] for _ in range(len(pa.solnMatrix)) ]
    for IDsm in range(len(pa.solnMatrix)):
        sm = pa.solnMatrix[IDsm]
        if sm.nrows() != sm.ncols():
            continue
        for ev in sm.eigenvectors_right():
            if ev[0] < zero:
                v = vector([ y.real() for y in  ev[1][0] ])
                scale = min([ abs(x) for x in v if abs(x) > 0.0001 ])
                guess = v/scale
                if roundable(guess):
                    rv = r(guess)
                    if rv*sm*rv > 0.01:
                        print("#",IDsm,rv,"Product was",rv*sm*rv)
                    else:
                        print(IDsm,rv," #check",rv*sm*rv)
                        zv[IDsm].append(rv)
                else:
                    print("#",IDsm,"FAIL ROUND with error ",roundable_error(guess),"for ",ev[0])
    return zv


def is_vector_almost_zero(v):
    if max([ abs(x) for x in v ]) < 0.000001:
        return True
    return False

def try_scale(v, sm, IDsm, verbose=True):
    if is_vector_almost_zero(v):
        return None
    
    scale = min([ abs(x) for x in v if abs(x) > 0.00009 ])
    for extrascale in [1,2,3,4,5,6]:
        guess = extrascale*v/scale
        #print(guess)
        if roundable(guess):
            rv = r(guess)
            if rv*sm*rv > 0.0001:
                if verbose:
                    print("#",IDsm,rv," failed check product was",rv*sm*rv)
                return None
            else:
                if verbose:
                    print("#",IDsm,rv," Got one, product",rv*sm*rv)
                return rv
    return None

def reduce_b_using_a(b, a, sm, IDsm):

    id_max_a = -1
    max_a = 0
    for num, id in zip(a, range(len(a))):
        if abs(num) > max_a:
            id_max_a = id
            max_a = abs(num)

    # Vector a is all 0
    # Means the vectors are the same and one could be dropped
    if max_a <= 0.0000001:
        return vector([0 for _ in b])
        #return b
    
    ba = b - (a/a[id_max_a])*b[id_max_a]
    rb = try_scale(ba, sm, IDsm)
    if rb != None:
        return rb
    return ba



def reduce_dirty_by_one_clean(dirty_ev, evc, sm, IDsm):
    # returns a list of new clean vectors
    new_clean = []
    indexes_to_remove = []
    
    for evd_id in range(len(dirty_ev)):
        dirty_ev[evd_id] = reduce_b_using_a(dirty_ev[evd_id], evc, sm, IDsm)
        if is_vector_almost_zero(dirty_ev[evd_id]):
            indexes_to_remove.append(evd_id)
        else:
            sev = try_scale(dirty_ev[evd_id], sm, IDsm)
            if sev != None:
                new_clean.append(sev)
                indexes_to_remove.append(evd_id)

    # Sort the indexes in descending order
    for i in sorted(indexes_to_remove, reverse=True):
        del dirty_ev[i]

    return new_clean
        


def reduce_dirty_by_dirty(dirty_ev, sm, IDsm):

    indexes_to_remove = []

    # returns a list of new clean vectors
    for evd_id_a in range(len(dirty_ev)):
        if evd_id_a in indexes_to_remove:
            continue

        for evd_id_b in range(evd_id_a+1,len(dirty_ev)):

            if evd_id_b in indexes_to_remove:
                continue

            dirty_ev[evd_id_b] = reduce_b_using_a(dirty_ev[evd_id_b], dirty_ev[evd_id_a], sm, IDsm)
            if is_vector_almost_zero(dirty_ev[evd_id_b]):
                indexes_to_remove.append(evd_id_b)
            else:
                sev = try_scale(dirty_ev[evd_id_b], sm, IDsm)
                if sev != None:
                    # Sort the indexes in descending order
                    indexes_to_remove.append(evd_id_b)
                    for i in sorted(indexes_to_remove, reverse=True):
                        del dirty_ev[i]
                    return sev

    for i in sorted(indexes_to_remove, reverse=True):
        del dirty_ev[i]

    return []
    

def guess_zeros_for_matrix(sm, IDsm=0, zero=1.0e-7, valid_digit=0.00001,verbose=False):
    if sm.nrows() != sm.ncols() or sm.nrows() == 1:
        return []
    
    evs=[]
    for ev in sm.eigenvectors_right():
        if ev[0] < zero:
            v = vector([ y.real() for y in  ev[1][0] ])
            scale = max([ abs(x) for x in v if abs(x) > valid_digit ])
            if scale > 0.00001:
                vs = v/scale
                evs.append(vs)
    
    print("Processing",IDsm,"  ",sm.nrows(),"X",sm.ncols(),"with",len(evs),"zeros")
    if len(evs) == 0:
        return []
    
    clean_ev=[]
    dirty_ev=[]
    
    for v in evs:
        sv = try_scale(v, sm, IDsm)
        if sv != None:
            print(IDsm,"Got vector",sv)
            clean_ev.append(sv)
        else:
            dirty_ev.append(v)


    if verbose:
        print("clean_ev:",len(clean_ev),clean_ev)
        print("dirty_ev:",len(dirty_ev),dirty_ev)

    evc_id = 0

    #print("AA starting with",len(dirty_ev))


    if len(dirty_ev) < 5:
        for dirty_evp in itertools.permutations(dirty_ev):
            dirty_ev_test = list(deepcopy(dirty_evp))
            maybe_clean = reduce_dirty_by_dirty(dirty_ev_test, sm, IDsm)
            if len(maybe_clean) != 0:
                #print("Found clean")
                clean_ev.append(maybe_clean)
                dirty_ev = list(dirty_ev_test)
                break
    else:
        for _ in range(50):
            dirty_ev_test = deepcopy(dirty_ev)
            maybe_clean = reduce_dirty_by_dirty(dirty_ev_test, sm, IDsm)
            if len(maybe_clean) != 0:
                #print("Found clean")
                clean_ev.append(maybe_clean)
                dirty_ev = dirty_ev_test
                break
            random.shuffle(dirty_ev)

    #print("XX")

    while evc_id < len(clean_ev):
        evc = clean_ev[evc_id]
        new_clean = reduce_dirty_by_one_clean(dirty_ev, evc, sm, IDsm)
        clean_ev += new_clean

        for _ in range(50):
            maybe_clean = reduce_dirty_by_dirty(dirty_ev, sm, IDsm)
            if len(maybe_clean) != 0:
                clean_ev.append(maybe_clean)
                break
            random.shuffle(dirty_ev)
        
        evc_id += 1

    #print("Left not cleaned",dirty_ev)
    if len(clean_ev) > 0:
        if len(dirty_ev) > 0:
            print("# ",IDsm," has ", len(clean_ev), " clean vectors and not cleaned",len(dirty_ev),"vectors")
        else:
            print("# ",IDsm," has ", len(clean_ev), " clean vectors, all cleaned.")
    else:
        if len(dirty_ev) > 0:
            print("# ",IDsm," has no clean vectors and not cleaned",len(dirty_ev),"vectors")



    return clean_ev

    #def guess_zero_vectors_eigenvectos_by_rounding(pa):
def guess_bunch_zeros(pa, zero=1.0e-7, product_tolerance=0.0000001):
    zv=[ [] for _ in range(len(pa.solnMatrix)) ]
    for IDsm in range(len(pa.solnMatrix)):
    #for IDsm in [0]:
        zv[IDsm] = guess_zeros_for_matrix(pa.solnMatrix[IDsm], IDsm, zero=zero)

    return zv

#zvv=guess_bunch_zeros(paa)




def guess_zero_eigenvectors(sm,IDsm=0, zero=1.0e-7,verbose=False):
    ''' Call as guess_zero_eigenvectors(p.solnMatrix[0])
    This is a fun utility that may work if the extremal construction
    is nicely balanced AND it is used on flags with k vertices and k-1 are labeled.
    It looks at zero eigenvectors, tries to find coordinates that are non-zero
    and in these coordinated it tries 0-1 vectors and sees if any are potentially
    zero vectors. Finally, it returns a basis of all these vectors.
    This is useful if there are possibly many ways to describe rooting or the
    extremal construction is not known.
    '''

    if sm.nrows() != sm.ncols() or sm.nrows() == 1:
        return []


    tightV = []

    for ev in sm.eigenvectors_right():
        if ev[0] > zero:
            continue

        print("Procesing vector")

        dimension = len(ev[1][0])
        z=vector([ round(1000*y) for y in ev[1][0] ] )
        zID=vector([ y for y in range(len(z)) if z[y] != 0 ] )

        # in zID we have locations of non-zeros in the extremal vector
        max_nonzeros = min([len(zID), 4])
        for subsetsize in range(max_nonzeros):
            for subset in itertools.combinations(zID, subsetsize):
                for nonzeros in itertools.product([-2,-1,1,2], repeat=subsetsize):
                    V = vector([0]*dimension )
                    for i,ID in enumerate(subset):
                        V[ID] = nonzeros[i]
                        value=V*sm*V
                        if V*sm*V < 0.0000001:
                            print('Found one with value',value,V)
                            tightV.append(vector(QQ, V))

    VV = QQ^dimension
    zv = VV.span(tightV)

    b=zv.basis()
    return b

def round_vector(x):
    for z in x:
        if abs(round(z)-z) > 0.001:
            print("Coordinate",z,"may be wrong")
    return vector([ round(z) for z in x])



def guess_zero_eigenvectors(sm, IDsm=0, zero=1.0e-7,verbose=False):
    ''' Call as guess_zero_eigenvectors(p.solnMatrix[0])
    This is a fun utility that may work if the extremal construction
    is nicely balanced AND it is used on flags with k vertices and k-1 are labeled.
    It looks at zero eigenvectors, tries to find coordinates that are non-zero
    and in these coordinated it tries 0-1 vectors and sees if any are potentially
    zero vectors. Finally, it returns a basis of all these vectors.
    This is useful if there are possibly many ways to describe rooting or the
    extremal construction is not known.
    '''

    if sm.nrows() != sm.ncols() or sm.nrows() == 1:
        return []

    tightV = []

    evs = sm.eigenvectors_right()
    x= [ y for y in evs if y[0] < zero ]

    ## Test how many zero vectors we have
    if len(x) == 0:
        return []

    b = sum( y[1][0] for y in x )
    z=vector([ round(100*real(y)) for y in b ] )
    zID=vector([ y for y in range(len(z)) if z[y] != 0 ] )


    print("Processing",IDsm,"  ",sm.nrows(),"X",sm.ncols(),"with",len(evs),"zeros got ",len(zID)," out of ",len(z)," coordinates non-zeros")

    dimension = len(z)

    # in zID we have locations of non-zeros in the extremal vector
    max_nonzeros = min([len(zID), 4])
    for subsetsize in range(max_nonzeros):
        for subset in itertools.combinations(zID, subsetsize):
            for nonzeros in itertools.product([-2,-1,1,2], repeat=subsetsize):
                V = vector([0]*dimension )
                for i,ID in enumerate(subset):
                    V[ID] = nonzeros[i]
                    value=V*sm*V
                    if V*sm*V < 0.0000001:
                        tightV.append(vector(QQ, V))
                        #print('Got ',len(tightV),'Found one with value',value,V)

    VV = QQ^dimension
    zv = VV.span(tightV)

    b=zv.basis()
    print('  Got ',len(b),'zero vectors')
    return b

def guess_bunch_zeros2(pa, zero=1.0e-7, product_tolerance=0.0000001):
    zv=[ [] for _ in range(len(pa.solnMatrix)) ]
    for IDsm in range(len(pa.solnMatrix)):
    #for IDsm in [0]:
        zv[IDsm] = guess_zero_eigenvectors(pa.solnMatrix[IDsm], IDsm, zero)

    return zv


def try_reduce(p, filename='altered.dat-s', zero=1.0e-7, zv=None):

    if zv==None:
        zv=guess_bunch_zeros(p, zero=zero)

    zero_vectors = sum([ len(v) for v in zv ])
    if zero_vectors == 0:
        print("No new zero vectors trying to get them in the other way ")

        zv=guess_bunch_zeros2(p, zero=zero)

    zero_vectors = sum([ len(v) for v in zv ])
    if zero_vectors == 0:
        print("No new zero vectors found ")
        return p
    

    with open(filename+'.zv.sage', 'w') as file:
        file.write("zv = " + repr(zv) + "\n")

    pa=p.change_base(zv)
    with open(filename+'.pm.pkl', "wb") as file:
         pickle.dump(p.projectionMatrices, file)

    pa=resolve(pa,filename)

    return pa



'''

with open('altered-dats.zv.sage', 'w') as file:
    file.write("zv = " + repr(zv) + "\n")

with open('alB.zv.sage', 'w') as file:
    file.write("zvA = " + repr(zvA) + "\n")



with open('pD_tightID.txt', 'w') as file:
    file.write("tightID = " + repr(pD.tightgraphsIDS) + "\n")


with open('alA.dat-s.pm.pkl', "wb") as file:
         pickle.dump(p.projectionMatrices, file)

with open('alB.dat-s.pm.pkl', "wb") as file:
         pickle.dump(pA.projectionMatrices, file)

with open('alC.dat-s.pm.pkl', "wb") as file:
         pickle.dump(pB.projectionMatrices, file)

with open('alD.dat-s.pm.pkl', "wb") as file:
         pickle.dump(pC.projectionMatrices, file)
                  

with open("alA.dat-s.pm.pkl", "rb") as file:
    pmA = pickle.load(file)         
with open("alB.dat-s.pm.pkl", "rb") as file:
    pmB = pickle.load(file)         
with open("alC.dat-s.pm.pkl", "rb") as file:
    pmC = pickle.load(file)         
with open("alD.dat-s.pm.pkl", "rb") as file:
    pmD = pickle.load(file)         
     
with open('alC.dat-s.pm.pkl', "wb") as file:
         pickle.dump(pmC, file)

with open('alD.dat-s.pm.pkl', "wb") as file:
         pickle.dump(pmD, file)
        

with open('combined.pm.pkl', "wb") as file:
         pickle.dump(pm, file)
                 

# The right projection matrix    
  ( pmD*pmC*pmB*pmA )

  
pa=resolve(pa,'all.dat-s')




'''

def combine_projections(projections):

    if len(projections) == 0:
        print("No projections provided")
        return []
    
    combinedp=[ None for _ in range(len(projections[0]))]

    for p in projections:
        for id,m in enumerate(p):
            if m != None:
                if combinedp[id] == None:
                    combinedp[id] = m
                else:
                    #if combinedp[id].nrows() != 0:
                        #print("Combining ID",id)
                        #print(m,"\n",combinedp[id])
                    if m.ncols() == combinedp[id].nrows():
                        combinedp[id] = m*combinedp[id]
                    else:
                        print("ERROR!!! at id",id)
    return combinedp




# Copies problem and does a base change based on zero vectors.
# Zero vectors are vectors we believe are zero eigenvectors of
# blocks in the problem matrix. If we find them all, it would
# be possible that CSDP returns to us a matrix that is positive
# definite instead of positive semidefinite and it becomes much
# easier to round.
#
def drop_blocks_with_zeros(p,zero=1.0e-7,):

    newproblem = deepcopy(p.problem);

    block_sizes=p.problem['block_sizes']
    blocks_to_delete=[]
    for idA in range(len(block_sizes)):
        if -block_sizes[idA] == p.problem['num_constraints']+1 or -block_sizes[idA] == p.problem['num_constraints']+2:
            continue
        #A = self.solnMatrix[idA]
        # Do not bother with matrices that are not squares
        # they do not need any special attention
        # Or if they are just constants...
        #if not A.is_square() or (1 == A.nrows() == A.ncols()):
        if block_sizes[idA] < 0:
            continue

        if  block_sizes[idA] == 1 or block_sizes[idA] == 0:
            continue

        A = p.solnMatrix[idA]
        zeros = sum([ 1 for  ev in A.eigenvalues() if ev < zero ])
        if zeros >= 1:
            blocks_to_delete.append(idA)

        #print "Done with",idA


    if len(blocks_to_delete) > 0:

        print("Deleting blocks",blocks_to_delete)

        # Sort the indexes in descending order
        for i in sorted(blocks_to_delete, reverse=True):
            del  newproblem['block_sizes'][i]
            del  newproblem['blocks'][i]

        newproblem['num_blocks'] = newproblem['num_blocks']  - len(blocks_to_delete)
    else:
        print("No blocks deleted")

    return newproblem

            
def drop_zeros_on_diagonal_blocks_and_resolve(p,filename='tmp_resolve.dat-s',zero=1.0e-7):

    zeros_removed = 0

    newproblem = deepcopy(p.problem);

    block_sizes=p.problem['block_sizes']
    blocks_to_delete=[]
    for idA in range(len(block_sizes)):
        if -block_sizes[idA] == p.problem['num_constraints']+1 or -block_sizes[idA] == p.problem['num_constraints']+2:
            continue

        # Only work on blocks that are not square
        if block_sizes[idA] >= 0:
            continue

        A = p.solnMatrix[idA]
        cols_to_delete = [ j for  j in range(A.ncols())  if A[0,j] < zero ]
        zeros_removed += len(cols_to_delete)

        if (len(cols_to_delete) == 0):
            continue

        if (len(cols_to_delete) == A.ncols()):
            print("Deleting entire block",idA)
            blocks_to_delete.append(idA)
            continue

        cols_to_keep = [j for j in range(A.ncols()) if j not in cols_to_delete]

        newproblem['block_sizes'][idA] = -len(cols_to_keep)
        for constraintID in range(len(newproblem['blocks'][idA])):
            newproblem['blocks'][idA][constraintID] = newproblem['blocks'][idA][constraintID][:, cols_to_keep]

    if len(blocks_to_delete) > 0:

        print("Deleting blocks",blocks_to_delete)

        # Sort the indexes in descending order
        for i in sorted(blocks_to_delete, reverse=True):
            del  newproblem['block_sizes'][i]
            del  newproblem['blocks'][i]

        newproblem['num_blocks'] = newproblem['num_blocks']  - len(blocks_to_delete)
    else:
        print("No blocks deleted")

    if zeros_removed == 0:
        return p

    print("Removed",zeros_removed,"zero coordinates from diagonal blocks, resolving")

    return  resolve(newproblem, filename=filename)

           